/***************************************************************** 
*  @file DX_ConnectivityTest.c
*  @brief A simple HOST-SeP connectivity test for 3rd Party 
*		  application. 3rd Party application may edit this file.
*
*  @author mottig
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/



/********************** Include Files *************************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "DX_VOS_Sem.h"
#include "SEPDriver.h" 
#include "ExtApp_op_code.h"
#include "gen.h"
#include "error.h"
#include "DX_ConnectivityTest.h"


/************************ Defines ******************************/
/************************ MACROS *******************************/
/************************ Global Data **************************/
/**************** Private function prototype *******************/




/************************ Public Functions ******************************/

/************************************************************************
 * 
 * @brief This function is used as a connectivity HOST-SeP test. It sends a message with 
 *		  one parameter only (counter variable) to the SeP and then gets acknowledge message 
 *		  back from SeP with this parameter (counter variable) increased by one. 
 *
 *        The function executes the following major steps:
 *
 *        1. Build a message with one parameter -this is the counter variable.
 *
 *        2. Sends message to SeP.
 *           
 *        3. Gets response message from SeP
 *        
 *		  4. Checks if the counter variable parameter increased by one.
 *
 *        5. Exit with appropriate error message.
 *
 * @return DxUint32 - On success DX_OK is returned
 */
EXPORT_C DxUint32 DxConnectivityTest()
{
   /* offset into SRAM */
   DxUint32_t   sramOffset;
   
   /* The return error identifier */
   DxUint32_t 	Error;

   /* parameters buffer : we have only one  */
   DxUint32_t   messageParam[1];
   
   /* op code */
   DxUint32_t   opCode;	   


   /* FUNCTION LOGIC */
   
   /* ............... local initializations .............................. */
   /* -------------------------------------------------------------------- */
   
   /* initializing the Error to O.K */
   Error = DX_OK;

   /* set offset to 0 */
   sramOffset = 0;
   
   /* set the op code*/
   opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_CONNECTIVITY_TEST_OP_CODE;

   /*--------------------------------
      start sending message to SEP 
   ---------------------------------*/
   
   /* initiate and start the message */
   Error = SEPDriver_BuildMsgHeader(&sramOffset, opCode);
   if(Error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* prepare message : set counter to 0 */
   messageParam[0] = 0;
   
   /* write message to shared memory */
   Error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                            sizeof(DxUint32_t),
                            sizeof(DxUint32_t),
                            &sramOffset,
                            DX_FALSE);
   if(Error != DX_OK)
   {
       goto end_function_unlock;
   }

   /* end message */
   SEPDriver_EndMessage(sramOffset);
            
   /* wait for the response */
   Error = SEPDriver_POLL_FOR_REPONSE();
   if(Error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Error = SEPDriver_CheckStatus(&sramOffset, opCode);
   
   if(Error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* read counter */
   Error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                            sizeof(DxUint32_t),
                            sizeof(DxUint32_t),
                            &sramOffset,
                            DX_FALSE);
                            
   if (Error != DX_OK) 
   {
   	 goto end_function_unlock;
   }

   /* check if counter was increased by one by SEP */
   if (messageParam[0] != 1)
   {
   	   Error = CONNECTIVITY_TEST_ERROR; 
   	   goto end_function_unlock;
   }
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Error;
   
   
}/* END OF DxConnectivityTest */   

